/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package sudoku;

import java.util.Arrays;

/**
 * The Sudoku playing field and methods to evaluate and manipulate it.
 * @author thomas.almy
 */
public class Field {
    /** Maximum value in square (minimum is 1, and 0 indicates vacant */
    public static final int MAXVAL = 9;
    /** Number of rows in playing field */
    public static final int ROWS = MAXVAL;
    /** Number of columns in playing field */
    public static final int COLUMNS = MAXVAL;
    /** Number of locations in playing field */
    public static final int LOCATIONS = ROWS*COLUMNS;
    /** Number of quadrants */
    public static final int QUADRANTS = MAXVAL;

    /** Bit array of possible values for this location, if not assigned. */
    public int [] possibles;
    /** Count of possible values for this location, if not assigned */
    public int [] possiblesCount;
    /** Playing location and value 1-9 assigned. Is 0 if not assigned. */
    public int [] squares;
    /** Location is locked from changes */
    private boolean [] _locked;
    /** Hint for location, one of NO_HINT, ONE_HINT, or TWO_HINT */
    private Hints [] _hintValue;
    /** Bit array of assigned values in each column */
    private int [] _columnMarks;
    /** Bit array of assigned values in each row */
    private int [] _rowMarks;
    /** Bit array of assigned values in each quadrant */
    private int [] _quadrantMarks;
    /* String equivalent of square values. Assumes MAXVAL=9 */
    private static final String [] _sqvals = {"- ", "1 ", "2 ", "3 ",
    "4 ", "5 ", "6 ", "7 ", "8 ", "9 "};

    @Override
    public String toString() {
        StringBuffer sb = new StringBuffer();
        for (int row = 0; row < ROWS; row++) {
            for (int col = 0; col < COLUMNS; col++) {
                sb.append(_sqvals[squares[coordinatesToIndex(col, row)]]);
            }
            sb.append("\n");
        }
        return sb.toString();
    }

    /**
     * Calculate the total number of hints (locations with values).
     * @return number of hints
     */
    public int hints() {
        int count = 0;
        for (int i = 0; i < LOCATIONS; i++) {
            if (squares[i] != 0) count++;
        }
        return count;
    }

    /**
     * Create a new playing Field with no locations assigned. CalculatePossibles should
     * be called before using.
     */
    public Field() {
        _locked = new boolean[LOCATIONS];
        _hintValue = new Hints[LOCATIONS];
        squares = new int[LOCATIONS];
        possibles = new int[LOCATIONS];
        possiblesCount = new int[LOCATIONS];
        _columnMarks = new int [COLUMNS];
        _rowMarks = new int [ROWS];
        _quadrantMarks = new int [QUADRANTS];
    }

    /**
     * Create a Field that is a copy of an existing Field.
     * @param f Field to copy
     */
    public Field(Field f) {
        this();
        System.arraycopy(f.squares, 0, squares, 0, LOCATIONS);
        System.arraycopy(f._hintValue, 0, _hintValue, 0, LOCATIONS);
        System.arraycopy(f._locked, 0, _locked, 0, LOCATIONS);
        System.arraycopy(f.possibles, 0, possibles, 0, LOCATIONS);
        System.arraycopy(f.possiblesCount, 0, possiblesCount, 0, LOCATIONS);
        System.arraycopy(f._columnMarks, 0, _columnMarks, 0, COLUMNS);
        System.arraycopy(f._rowMarks, 0, _rowMarks, 0, ROWS);
        System.arraycopy(f._quadrantMarks, 0, _quadrantMarks, 0, QUADRANTS);
    }

    /**
     * Convert a location number to a row number
     * @param index location
     * @return row
     */
    static public int indexToRow(int index) {
        return index / COLUMNS;
    }

    /**
     * Convert a location number to a column number
     * @param index location
     * @return column
     */
    static public int indexToColumn(int index) {
        return index % COLUMNS;
    }

    /**
     *  Convert a location number to a quadrant number
     * @param index location
     * @return quadrant
     */
    static public int indexToQuadrant(int index) {
        return (indexToColumn(index)/3)*3 + indexToRow(index)/3 ;
    }

    /**
     * Convert row and column numbers to a location number
     * @param x column
     * @param y row
     * @return location
     */
    static public int coordinatesToIndex(int x, int y) {
        return y*COLUMNS + x;
    }

    /**
     * Fill in a location. Result is valid only if originally unfilled and new value
     * is 1-9. Otherwise call setMarks to correct mark values.
     * @param index location to fill
     * @param value value to fill location with
     */
    public void fillAt(int index, int value) {
        squares[index] = value;
        int mask = 1 << (value-1);
        _columnMarks[indexToColumn(index)] |= mask;
        _rowMarks[indexToRow(index)] |= mask;
        _quadrantMarks[indexToQuadrant(index)] |= mask;
    }

    /**
     * Correct the mark values after changing or removing a value in a location.
     */
    public void setMarks() {
        // Reset marks;
        _columnMarks = new int [COLUMNS];
        _rowMarks = new int [ROWS];
        _quadrantMarks = new int [QUADRANTS];
        for (int i = 0; i < LOCATIONS; i++)
        {
            fillAt(i, squares[i]);
        }
    }


    /**
     * Either locks locations that have values or unlocks all locations
     * @param flag true to lock, false to unlock
     */
    public void lockArray(boolean flag) {
        int i;

        if (flag) {
            // Lock cells and set marked array
            for (i = 0; i < LOCATIONS; i++) _locked[i] = squares[i] != 0;
        } else {
            Arrays.fill(_locked, false);
        }
    }

    /**
     * Fills in possibles, possiblesCount, and _hintValues arrays based.
     * Execute after fillAt/setMarks and before determining next move.
     */
    public void calculatePossibles() {
        int maxPossible = MAXVAL+1;
        for (int i = 0; i < LOCATIONS; i++) {
            if (squares[i] != 0) {
                possibles[i] = 0;
                possiblesCount[i] = 0;
            } else {
                final int row = indexToRow(i);
                final int col = indexToColumn(i);
                final int quad = indexToQuadrant(i);
                int possible = 0x1ff & ~(_columnMarks[col] | _rowMarks[row] | _quadrantMarks[quad]);
                int count = 0;

                possibles[i] = possible;
                while (possible != 0) { // count 1 bits
                     count += possible & 1;
                     possible >>= 1;
                 }
                 possiblesCount[i] = count;
                 if (count > 0 && count < maxPossible) maxPossible = count;
            }
        }
        if (maxPossible == 1 || maxPossible == 2) {
            Hints val = maxPossible == 2 ? Hints.TWO_HINT : Hints.ONE_HINT;
            for (int i = 0; i < LOCATIONS; i++) {
                if (squares[i] == 0 && possiblesCount[i] == maxPossible) {
                    _hintValue[i] = val;
                } else {
                    _hintValue[i] = Hints.NO_HINT;
                }
            }
        } else {
            Arrays.fill(_hintValue, Hints.NO_HINT);
        }
    }

    /**
     * Examine current field to determine if there are available moves.
     *
     * @return SUCCESS if the field is completely filled, FAIL if not filled
     * but there are no possible moves, or IN_PROGRESS if there are possible moves.
     */
    public GameState getState() {
            calculatePossibles();
            int bestPos = 0, bestVal = MAXVAL+1, bestCount = 0;
            for (int i = 0; i < LOCATIONS; i++) {
                if (squares[i] == 0) { // Not filled in
                    if (possiblesCount[i] < bestVal) { // better choice
                       bestPos = i;
                       bestCount = 0;
                       bestVal = possiblesCount[i];
                    } else if (possiblesCount[i] == bestVal) { // equal choice
                     bestCount++;
                    }
                }
            }
            if (bestVal == MAXVAL+1) return GameState.SUCCESS;
            if (bestVal == 0) return GameState.FAIL;
            return GameState.IN_PROGRESS;

    }

    /**
     * A locked location is presumably one that was initially hinted. Locking
     * changes the way the location is to be displayed, and locked locations
     * locations cannot be changed.
     *
     * @param i location index
     * @return locked flag
     */
    public boolean isLocked(int i) {
        return _locked[i];
    }

    /**
     * The hint value is one of NO_HINT, ONE_HINT, or TWO_HINT and should be
     * displayed if hints are enabled. Hint values are calculated in calculatePossibles.
     * @param i location index
     * @return NO_HINT, ONE_HINT, or TWO_HINT
     */
    public Hints hintValue(int i) {
        return _hintValue[i];
    }
}
