package sudoku;

import java.util.ArrayList;
import java.util.Collections;

/**
 * The guts of Sudoku. The PlayEngine is capable of finding correct moves.
 * @author thomas.almy
 */
public class PlayEngine {

    private Field _f;
    private boolean _noBacktrack = false;
    private boolean _checkMultiples = false;

    /**
     * Call this method if PlayEngine is to disallow backtracking. This means that
     * the puzzle must be solvable with only obvious moves.
     */
    public void setNoBacktrack() {
        _noBacktrack = true;
    }

    /** Call this method with an argument of true to have the PlayEngine check for
     * multiple solutions and give an exception if found. We only want to create
     * puzzles with unique solutions, I guess.
     * @param check
     */
    public void setCheckMultiples(boolean check) {
        _checkMultiples = check;
    }

    /**
     * Create a PlayEngine to play the given field.
     * @param field
     */
    public PlayEngine(Field field) {
        _f = field;
    }

    /**
     * Play out the argument field to see if it is solvable. An EASY_SOLUTION has
     * only obvious moves (no backtracking necessary).
     * @param f
     * @return NO_SOLUTION, EASY_SOLUTION, or HARD_SOLUTION
     */
    public static SolutionType verify(Field f) {
        // First decide if we even need to play it
        GameState state = f.getState();
        switch (state) {
            case FAIL:
                return SolutionType.NO_SOLUTION;
            case SUCCESS:
                return SolutionType.EASY_SOLUTION;
            default:
                break;
        }

        // Try playing without backtracking
        Field new_f = new Field(f);
        PlayEngine p = new PlayEngine(new_f);
        p.setNoBacktrack();
        try {
            new_f = p.playUntilSolvedOrQuit();
        } catch (MultipleSolutionException ex) {
        }
        if (new_f == null) {
            return SolutionType.NO_SOLUTION;
        } else {
            state = new_f.getState();
            switch (state) {
                case FAIL:
                    return SolutionType.NO_SOLUTION;
                case SUCCESS:
                    return SolutionType.EASY_SOLUTION;
                default:
                    break;
            }
        }

        // If that didn't work, then go for broke
        new_f = new Field(f);
        p = new PlayEngine(new_f);
        try {
            new_f = p.playUntilSolvedOrQuit();
        } catch (MultipleSolutionException ex) {
        }

        if (new_f == null) {
            return SolutionType.NO_SOLUTION;
        } else {
            return SolutionType.HARD_SOLUTION;
        }


    }


    /**
     * Main playing routine. Plays until solved, depending on backtracking if
     * enabled.
     * @return Solved field, if one exists, or null if none exists.
     * @throws MultipleSolutionException if checkMultiples(true) called first and
     * there is more than one solution possible.
     */
    public Field playUntilSolvedOrQuit() throws MultipleSolutionException {
        _f.calculatePossibles();
        while (true) { // Iterate on moves
            // find the best move
            int bestPos = 0, bestVal = Field.MAXVAL+1, bestCount = 0;
            for (int i = 0; i < Field.LOCATIONS; i++) {
                if (_f.squares[i] == 0) { // Not filled in
                    if (_f.possiblesCount[i] < bestVal) { // better choice
                        bestPos = i;
                        bestCount = 0;
                        bestVal = _f.possiblesCount[i];
                    } else if (_f.possiblesCount[i] == bestVal) { // equal choice
                        bestCount++;
                    }
                }
            }
            if (bestVal == Field.MAXVAL+1) {
                return _f; // All filled in -- success
            }
            if (bestVal == 0) {
                return null; // One square has no possible value -- fail
            }
            if (bestVal == 1) { // Square has only one choice -- great news
                int choice = 0;
                switch (_f.possibles[bestPos]) {
                    case 1:
                        choice = 1;
                        break;
                    case 2:
                        choice = 2;
                        break;
                    case 4:
                        choice = 3;
                        break;
                    case 8:
                        choice = 4;
                        break;
                    case 16:
                        choice = 5;
                        break;
                    case 32:
                        choice = 6;
                        break;
                    case 64:
                        choice = 7;
                        break;
                    case 128:
                        choice = 8;
                        break;
                    case 256:
                        choice = 9;
                        break;
                    default:
                        System.out.println("No valid choice in " + _f.possibles[bestPos]);
                }
                _f.fillAt(bestPos, choice);
                _f.calculatePossibles();
            } else {

                if (_noBacktrack) { // Backtracking disabled -- return best state
                    return _f;
                }

                // We must try all alternatives
                int possibles = _f.possibles[bestPos];
                int trial = 1;
                int alternatives = 0;
                Field newField = null;

                do {
                    if ((possibles & (1 << (trial - 1))) != 0) {
                        newField = new Field(_f); // Make a copy of current field
                        newField.fillAt(bestPos, trial); // And set trial
                        PlayEngine p = new PlayEngine(newField); // Make a new engine to play it
                        p.setCheckMultiples(_checkMultiples);
                        newField = p.playUntilSolvedOrQuit();
                        if (newField != null) {
                            if (_checkMultiples) {
                                alternatives++;
                                if (alternatives > 1) throw new MultipleSolutionException();

                            } else {
                                return newField; // We have a winner!
                            }
                        }
                    }
                    trial++; // Try next value
                } while (trial < Field.MAXVAL+1);
                if (_checkMultiples && alternatives == 1) {
                    // If no multiples just return a field since we don't care
                    // about the actual value
                    return new Field();
                }
                return null; // None of the choices worked
            }
        }
    }

    /**
     * Alternative solver that makes random choices at random locations to create
     * a new solution, typically from an empty field. Never checks for multiple
     * solutions since the goal is to find a single random solution.
     * @return solved field or null if none found.
     */
    public Field playRandomlyUntilSolvedOrQuit() {
        _f.calculatePossibles();
        while (true) { // Iterate on moves
            // find the best move
            int bestVal = Field.MAXVAL+1;
            ArrayList<Integer> choices = new ArrayList<Integer>();
            for (int i = 0; i < Field.LOCATIONS; i++) {
                if (_f.squares[i] == 0) { // Not filled in
                    if (_f.possiblesCount[i] < bestVal) { // better choice
                        choices.clear();
                        choices.add(Integer.valueOf(i));
                        bestVal = _f.possiblesCount[i];
                    } else if (_f.possiblesCount[i] == bestVal) { // equal choice
                        choices.add(Integer.valueOf(i));
                    }
                }
            }
            if (bestVal == Field.MAXVAL+1) {
                return _f; // All filled in -- success
            }
            if (bestVal == 0) {
                return null; // One square has no possible value -- fail
            }
            if (bestVal == 1) { // Square has only one choice -- great news
                int choice = 0;
                Collections.shuffle(choices, Chooser.getChooser().getRandom());
                int bestPos = choices.get(0);
                switch (_f.possibles[bestPos]) {
                    case 1:
                        choice = 1;
                        break;
                    case 2:
                        choice = 2;
                        break;
                    case 4:
                        choice = 3;
                        break;
                    case 8:
                        choice = 4;
                        break;
                    case 16:
                        choice = 5;
                        break;
                    case 32:
                        choice = 6;
                        break;
                    case 64:
                        choice = 7;
                        break;
                    case 128:
                        choice = 8;
                        break;
                    case 256:
                        choice = 9;
                        break;
                    default:
                        System.out.println("No valid choice in " + _f.possibles[bestPos]);
                }
                _f.fillAt(bestPos, choice);
                _f.calculatePossibles();
            } else {

                if (_noBacktrack) { // Backtracking disabled -- return best state
                    return _f;
                }

                // We must try all alternatives
                int bestPos = (Integer)choices.get(Chooser.getChooser().nextInt(choices.size()));
                int possibles = _f.possibles[bestPos];

                choices.clear();
                for (int i = 0; i < Field.MAXVAL; i++ ) {
                    if ((possibles & (1 << i)) != 0) {
                        choices.add(Integer.valueOf(i+1));
                    }
                }
                
                for (Object trialObj : choices) {
                    int trial = (Integer)trialObj;
                    Field newField = new Field(_f); // Make a copy of current field
                    newField.fillAt(bestPos, trial); // And set trial
                    PlayEngine p = new PlayEngine(newField); // Make a new engine to play it
                    newField = p.playRandomlyUntilSolvedOrQuit();
                    if (newField != null) {
                        return newField; // We have a winner!
                    }
                }
                return null; // None of the choices worked
            }
        }
    }


}

