/* ECHOREPT is Copyright 1993 by Thomas Almy.
   It may be used without charge.
   It may be distributed provided no charge is made other than
   actual distribution costs.  */

/* Compile with Borland C, Compact memory modem (-mc). */


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <alloc.h>

#define NAMESIZE (31)
#define MAXHUD (200)

char *groups = 0;		/* new groups feature */
char dbname[12] = "netrep2.dat";

char gepath[128];

FILE *dbfp;

int thisdate;
struct tm *block;

int dbsize;

typedef struct   {
	char echoname[NAMESIZE];
    int area;       /* area number */
	int tally[30];	/* 30 day record */
} DBRECORD;

DBRECORD huge *db;  /* the records */
DBRECORD huge **dbp;	/* array of pointers to sorted records */

#define a_area ((*(DBRECORD **)a)->area)
#define b_area ((*(DBRECORD **)b)->area)
#define a_name ((*(DBRECORD **)a)->echoname)
#define b_name ((*(DBRECORD **)b)->echoname)

int sort_function(const void *a, const void *b) {
	if (a_area == 0) return (b_area ? 1 : 0);
	if (b_area == 0) return -1;
	if (a_area <= MAXHUD) { /* first is hudson */
		if (b_area <= MAXHUD)
			return (a_area < b_area? -1 : 1);	/* no dupes are possible */
		else
			return -1; /* Hudson less than passthrough */
	}
	else { /* first is passthrough */
		if (b_area <= MAXHUD)
			return 1;
		else 
			return strcmp(a_name, b_name);
	}
}

void sortit(void) {
    int i, i2, j;
	for (i=0; i<dbsize; i++) {	/* delete entries that have no messages */
		for (j=0,i2=0; i2<30; i2++) j += db[i].tally[i2];
		if (j == 0) { /* clear entry */
			db[i].echoname[0] = 0;	
			db[i].area = 0;
		}
	}
	qsort((void *)dbp, dbsize, sizeof(DBRECORD *), sort_function);
}

void makereport(char *filename, int days, char *title) {
	
	FILE *fp;
	char ans[64];
	char areastr[6];
	int st = (days>28 || days==1 ? 0 : 1);	/* starting day */
	int i,j;
	long gt=0;	/* grand total */
	int tot;	/* total for each area */
	int wtot[4];/* totals for weeks */

	days += st;	/* when we start on day 1, end is one further */
	
	if ((fp = fopen(filename, "w")) == NULL) {
		printf( "Error--unable to create %s\n", filename);
		return;	/* no error exit on this one */
	}
	
	strcpy(ans, asctime(block));
	if (groups != NULL) {
		fprintf(fp, "%s          Report prepared %.10s, %.4s\nfor group", 
				title, ans, &ans[20]);
		fprintf(fp, groups[1]!= '\0'? "s %s\n\n" : " %s\n\n", groups);
	}
	else
		fprintf(fp, "%s          Report prepared %.10s, %.4s\n\n", 
				title, ans, &ans[20]);

	if (days==8) 
		fputs(" #  Area Name                Daily Totals            Total Percent\n",fp);
	else if (days==29)
		fputs(" #  Area Name                Weekly Totals   Total Percent\n",fp);
	else
		fputs(" #  Area Name                      Total Percent\n",fp);

	for (i = 0; i < dbsize; i++)  /* calculate grand total messages */
		for (j = st; j < days; j++) 
			gt += db[i].tally[j];

	for (i = 0; i < dbsize; i++) {
		tot = 0;
		for (j = st; j < days; j++) tot += dbp[i]->tally[j];
		if (tot == 0) continue;
		if (dbp[i]->area > MAXHUD) 
			strcpy(areastr, " P ");
		else
			sprintf(areastr, "%3d", dbp[i]->area);
		if (days==8) { /* different for weekly report */
			fprintf(fp, "%s %-20.20s%4d%4d%4d%4d%4d%4d%4d %5d %.1f%%\n", 
				areastr,
				dbp[i]->echoname, 
				dbp[i]->tally[7],
				dbp[i]->tally[6],
				dbp[i]->tally[5],
				dbp[i]->tally[4],
				dbp[i]->tally[3],
				dbp[i]->tally[2],
				dbp[i]->tally[1],
				tot, 
				(100.0 * tot)/gt);
		}
		else if (days==29) { /* and for monthly report */
			wtot[0] = wtot[1] = wtot[2] = wtot[3] = 0;
			for (j=1; j<8; j++) {
				wtot[0] += dbp[i]->tally[j];
				wtot[1] += dbp[i]->tally[j+7];
				wtot[2] += dbp[i]->tally[j+14];
				wtot[3] += dbp[i]->tally[j+21];
			}
			fprintf(fp, "%s %-20.20s%4d %4d %4d %4d  %5d  %.1f%%\n",
				areastr,
				dbp[i]->echoname,
				wtot[3], wtot[2], wtot[1], wtot[0],
				tot,
				(100.0 * tot)/gt);
		}
		else {
			fprintf(fp, "%s %-30.30s %5d %.1f%%\n", areastr,
				dbp[i]->echoname, 
				tot, 
				(100.0 * tot)/gt);
		}
	}
	fprintf(fp, "\n%ld total messages\n", gt);
	fclose(fp);
}

void readsummary(void) {
	FILE *transp;
	char filename[128];
	char buff[128];
	char area[NAMESIZE];
	char gr;
	int i,j, areanum;

	strcpy(filename, gepath);
	strcat(filename, "summary.log");
	
	if ((transp = fopen(filename, "r")) == NULL) {
		printf( "Note-- summary.log file not found\n");
		return;
	}
	
	i=5;
	while (i--) fgets(buff, sizeof(buff), transp); /* header lines */
	
	while (fgets(buff, sizeof(buff), transp) != NULL) {
		if (strncmp(buff, "-----", 5) == 0) break;	/* end reached */
		if (sscanf(buff, "%d %30s %d %*d %c", &areanum, &area, &j, &gr) != 4 ||
			areanum < 1 ) continue; /* bad record ?? */
		if (groups != NULL && strchr(groups, gr) == NULL)
			continue; /* not in this group */
		for (i=0; i<dbsize && db[i].echoname[0] != 0; i++) 
			if (strcmp(area, db[i].echoname)==0) break;
		if (i < dbsize) { /* new record -- if there is room */
			strcpy(db[i].echoname, area);
			db[i].area = areanum;
			db[i].tally[0] += j;
		}
	}
	fclose(transp);
	if (groups == NULL)
		unlink(filename);	/* get rid of file if not a partial scan */
}
	

void getdb(void) {	/* get the database */
	time_t timer;
	int olddate;
	int i, diff;
	int actual_size;

	timer = time(NULL);
	block = localtime(&timer);
	thisdate = block->tm_yday;
	
	if ((dbfp = fopen(dbname,"rb")) == NULL) { /* no file */
		actual_size = 0;
	}
	else {	/* file exists */
		setvbuf(dbfp, NULL, _IOFBF, 8192);
		fseek(dbfp, 0L, SEEK_END);
		actual_size = (int) ((ftell(dbfp) - sizeof(int)) / sizeof(DBRECORD));
		rewind(dbfp);
	}

	dbsize=(int)((farcoreleft()-10000)/(sizeof(DBRECORD)+sizeof(DBRECORD*)));

	if ((db = farcalloc(dbsize, sizeof(DBRECORD))) == NULL ||
		(dbp = farcalloc(dbsize, sizeof(DBRECORD *))) == NULL) {
		printf( "Error -- insufficient memory\n");
		exit(1);
	}

	for (i=0; i<dbsize; i++) dbp[i] = &db[i];

	if (actual_size==0) return;

	fread(&olddate, sizeof(int), 1, dbfp);
	
	for (i=0; i < actual_size; i++)
		if (fread(&db[i], sizeof(DBRECORD), 1, dbfp) != 1) {
			printf( "Error -- bad file\n", dbname);
			exit(1);
		}

	fclose(dbfp);
	
	if (olddate == thisdate) return;
	
	/* Date changed -- shift the counter array for each echo by the
	   number of days that have passed */

	if (olddate > thisdate) {
		olddate -= 365;
		if (block->tm_year % 4 == 1) olddate--;
	}
	
	diff = thisdate - olddate;

	if (diff > 29) { /* boy has time passed! */
		diff = 30;
	}

	for (i=0; i<dbsize; i++) {
		memmove(&db[i].tally[diff], &db[i].tally[0], sizeof(int)*(30-diff));
		memset(&db[i].tally[0], 0, sizeof(int)*diff);
	}
	return;
}
	

void badwrite(void) {
		printf( "Error -- write of %s failed", dbname);
		exit(1);
}

void putdb(void) {
	int i=0;
	
	if ((dbfp = fopen(dbname,"wb")) == NULL) { /* can't open output */
		printf("Error -- can't write %s file", dbname);
		exit(1);
	}

	setvbuf(dbfp, NULL, _IOFBF, 8192);

	if (fwrite(&thisdate, sizeof(int), 1, dbfp) != 1) badwrite();

	while (dbp[i]->area != 0 && i < dbsize) 
		if (fwrite(dbp[i++], sizeof(DBRECORD), 1, dbfp) != 1) badwrite();

	fclose(dbfp);
}


void main(int argc, char **argv) {
	char *cp;
	
	if (argc > 1 && argv[1][0] == '-')	{ /* group selection provided */
		groups = argv[1] + 1;
		strncpy(dbname+8, groups, 3);	/* new name extension */
		argc--;
		argv++;
	}

	strcpy(gepath, getenv("GE"));	/* gecho path */
	if ((cp = strchr(gepath, ' '))!=NULL) 
		*cp = '\0'; /* ignore color argument */
	if (*gepath && gepath[strlen(gepath)-1] !='\\') strcat(gepath, "\\");

	getdb();
	
	readsummary();	/* read summary file, if any */

	if (argc > 1) sortit();	/* sort database if we are making a report */
	if(argc > 1) makereport(argv[1], 1,  "Todays's Echo Traffic    ");
	if(argc > 2) makereport(argv[2], 7,  "Previous Week's Echo Traffic");
	if(argc > 3) makereport(argv[3], 28, "Four Week Echo Traffic      ");
	if(argc > 4) makereport(argv[4], 30, "Thirty Day Echo Traffic     ");
	
	putdb();
}
