/* Jelbum -- Jumble (tm) Puzzle Solver */
/* Written by Tom Almy  10/89 */
/* Public Domain */


/* Compile with "Compact" model -- small code, large data */
/* Tested with Turbo-C 2.0 and Microsoft C 5.0 (which is faster) */

/* use this version for best performance */

/* To solve a single jumbled word, invoke as:
	jelbum jelbum

   To get the cartoon solution when there are two words (or three if one
   is "A", in which case toss the letter and treat as two word solution):
	   
   jelbum jumbelovels -5
	   
   Where the number is the size of one of the words (faster if smaller
   word size is given.
	   
   You can also say:
	   
   jelbum jumbelovels 6
	   
   Which will, in this case, list all 6 letter words that can be made from
   the 11 letters, along with the letters that didn't match. */


#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <fcntl.h>

struct datastruct {
	int tag;
	char string[100];
};

struct linkeddata {
	int tag;
	struct linkeddata *next;
    int thisStat[26];
	char string[1];
};

static int currentFile = 0;

static char *FILES[] = {"","","size2.dic","size3.dic","size4.dic",
"size5.dic","size6.dic","size7.dic","size8.dic","size9.dic","size10.dic"};

static int recordSize = 0;	/* size of record in file */
static int recsPerBuf = 0;	/* number of records per buffer */
static int recsInLastBuf = 0;
static int recsInThisBuf = 0;
static int curRec = 0;		/* current record number */
static int lastBuf = 0;		/* last buffer */
static int curBuf = 0;		/* current buffer number */
static struct datastruct *wordBufs[20];

void initialize(int i)
{
	int infile;

	if (currentFile == i) {
		curBuf = 0;
		curRec = 0;
		recsInThisBuf = (curBuf == lastBuf ? recsInLastBuf : recsPerBuf);
		return;
	}
	if (currentFile != 0) { /* free used memory */
		for (curBuf =0; curBuf <= lastBuf; curBuf++)
			free(wordBufs[curBuf]);
	}

	currentFile = i;
	if ((infile=open(FILES[i],O_RDONLY)) == 0) {
		fprintf(stderr, "ERROR: file %s not found\n");
		exit(1);
	}
	recordSize = sizeof(int)+i;
	recsInThisBuf = 32767 / recordSize;
	recsPerBuf = recsInThisBuf;
	lastBuf = curBuf = curRec = 0;
	while (1) {
		if ((wordBufs[lastBuf] = malloc(recordSize*recsPerBuf)) == NULL) {
			fprintf(stderr,"Insufficient memory\n");
			exit(1);
		}
/*		recsInLastBuf = fread(wordBufs[lastBuf],recordSize,recsPerBuf,infile); */
		recsInLastBuf = ((unsigned int)read(infile,
			wordBufs[lastBuf],recordSize*recsPerBuf))/recordSize;
		if (recsInLastBuf == 0) { /* one too many buffers */
			free(wordBufs[lastBuf--]);
			recsInLastBuf = recsPerBuf;
			break;
		}
		if (recsInLastBuf < recsPerBuf) break;
		lastBuf++;
	}
	close(infile);
	if (lastBuf == 0) recsInThisBuf = recsInLastBuf;
}

static struct datastruct *data;
	
static int searchStat[26];
static int searchLen,recLen;
static int searchValue;

setSearchFromStat(int *srchbuf, int size)
{
	int i;
	searchLen = size;
	recLen = size + sizeof(int);
	memcpy(searchStat,srchbuf,sizeof(searchStat));

	for (searchValue = 0, i = 0; i < 26; i++) 
		searchValue += searchStat[i]*i;

	initialize(size);
	return (0);
}


int setSearch(char *s)	/* return non-zero if arg string is bad */
{
	int i,c;
	searchLen = strlen(s);
	recLen = searchLen + sizeof(int);
	if (searchLen < 2 || searchLen > 10) return (1);
	memset(searchStat,0,sizeof(searchStat));
	for (searchValue = 0, i = 0; i < searchLen; i++) {
		if (islower(s[i]) == 0) return(1);
		c = s[i] - 'a';
		searchValue += c;
		searchStat[c] += 1;
	};
	initialize(searchLen);
	return (0);
}
		


char *doSearch() /* search for matching string */
{
	int thisStat[26];
	int i;
	while (1) {

		if (recsInThisBuf == curRec) { /* go to next buffer */
			curRec = 0;
			curBuf++;
			if (curBuf > lastBuf) return NULL;
			if (curBuf == lastBuf) recsInThisBuf = recsInLastBuf;
		}
		data = (struct datastruct *)(((long)wordBufs[curBuf])+recordSize*curRec);
		curRec++;
		
		if (searchValue == data->tag) {
			memcpy(thisStat,searchStat,sizeof(thisStat));
			for (i=0; i<searchLen; i++) 
				if (--thisStat[data->string[i]-'a'] < 0) goto thisFail;
			return data->string;
		}
		thisFail:;
	}
	return NULL;
}


void simplejumble(char *s)
{
	char *ptr;
	if (setSearch(s)) {
		fprintf(stderr,"%s is an invalid string for Jumble\n",s);
		exit(1);
	}
	while ((ptr = doSearch()) != NULL) {
		printf("%.*s\n",strlen(s),ptr);
	}
}

static int ourStat[26];
static int ourInitStat[26];
static int maxcnt;
static int complete;
static struct linkeddata *datalist;
static int passnum;

void domessy(int cnt, int start)
{
	int i;
	int j;
	int ii;
	char *s;
	struct linkeddata *n;

	for (i = start; i < 26; i++) {
		if (ourStat[i] != 0) {
			ourStat[i]--;
			if (cnt == 1) { /* process it now! */
				setSearchFromStat(ourStat,maxcnt);
				if (complete) {
					fprintf(stderr,"Pass %d\r", passnum++);
				}
				while ((s=doSearch()) != NULL) {
					if (complete) {
						n = malloc(sizeof(struct linkeddata)+maxcnt);
						n->tag = data->tag;
						memcpy(n->string,data->string,maxcnt);
						for (j = 0; j< 26; j++) {
							n->thisStat[j] = ourInitStat[j] - ourStat[j];
						}
						n->next = datalist;
						datalist = n;
					}
					else {
						printf("%.*s with:",maxcnt,s);
						for (j = 0; j < 26; j++) {
							ii = ourInitStat[j] - ourStat[j];
							while (ii-- > 0) printf("%c ",j+'a');
						}
						printf("\n");
					}
				}
			}
			else { /* must recurse */
				domessy(cnt-1,i);
			}
			ourStat[i]++;
		}
	}
}

void messyjumble(char *s, int i)
{
	int len = strlen(s);
	int j;
	char *ptr;
	static struct linkeddata *dl;

	datalist = NULL;

	if (i < 0) {	/* run to completion */
		complete = 1;
		i = -i;
	}
	else complete = 0;
	
	maxcnt = i;
	
	if (i < 2 || i > 10 || i >= len) {
		fprintf(stderr,"count value out of range");
		exit(1);
	}
	
	memset(ourStat,0,sizeof(ourStat));
	for (j= 0; j < len; j++) {
		if (islower(s[j]) == 0) {
			fprintf(stderr,"%s is invalid string for jumble");
			exit(1);
		}
		ourStat[s[j]-'a'] += 1;
	}

	memcpy(ourInitStat,ourStat,sizeof(ourStat));

	passnum = 1;		/* give user some feedback */
	domessy(len-i, 0);

	if (complete) {
		j = 0;
		dl = datalist;
		while (dl != NULL) {
			j++;
			dl = dl->next;
		}
		if (j == 0) {
			fprintf(stderr,"\nNo matches found\n");
			return;
		}
		
		fprintf(stderr,
			"\n%d matches found\nNow searching for second words...\n",j);
		while (datalist != NULL) {
			setSearchFromStat(datalist->thisStat, len-i);
			while ((ptr = doSearch()) != NULL) {
				printf("%.*s %.*s\n",i, datalist->string, len-i, ptr);
			}
			datalist = datalist->next;
		}
	}
			

}


void main(int argc, char **argv)
{
	char *ptr;
	int count;
	
	if (argc < 2 || argc > 3) {
		fprintf(stderr,"Jumble program\nUsage: jumble word\n or jumble word count\n");
		exit(1);
	}

	if (argc == 2) simplejumble(argv[1]);
	else messyjumble(argv[1],atoi(argv[2]));


}
