;;; Enable use of ERRSET function:
(nodebug)
(defparameter test-number 0)
(defparameter bugs-found 0)

(defvar *abort-on-error* t)	; Abort tests on error, otherwise continue
(defvar *verbose-tests* t)	; Print message on successful tests too.

;;; Define a central error-report fn:
(defun r-fail (i x why)
  (setq bugs-found (1+ bugs-found))
  (format t "***** Failed test number ~s: ~s~%  because: ~a~%" i x why)
  (when *abort-on-error* (top-level)))

(defun passed-test (i x) 
       (when *verbose-tests* 
	     (format t "Passed test ~s: ~s~%" i x)))

;;; Squeeze out some redundancy:
(defmacro should-fail (x)
  (setq test-number (1+ test-number))
  `(if (errset ,x nil) 
       (r-fail test-number ',x "execution didn't fail")
       (passed-test test-number ',x)))
(defmacro should-work (x)
  (setq test-number (1+ test-number))
  `(if (not (errset ,x nil)) 
       (r-fail test-number ',x "execution failed")
       (passed-test test-number ',x)))
(defmacro should-be-true (x &aux (y (gensym)))
  (setq test-number (1+ test-number))
  `(let ((,y (errset ,x nil))) 
       (cond ((null ,y) (r-fail test-number ',x "execution failed"))
	     ((eq (car ,y) 'nil) (r-fail test-number ',x "returned NIL"))
	     (t (passed-test test-number ',x)))))
(defmacro should-be-false (x &aux (y (gensym)))
  (setq test-number (1+ test-number))
  `(let ((,y (errset ,x nil))) 
       (cond ((null ,y) (r-fail test-number ',x "execution failed"))
	     ((not (eq (car ,y) 'nil)) 
	      (r-fail test-number ',x (format nil "returned ~s" (car ,y))))
	     (t (passed-test test-number ',x)))))

;; Compare contents of a collection to a (must be sorted) list
(defun cmp (x y &optional clss)
       (and (if clss
                (eq (send x :class) clss)
                t)
            (equal (sort (send x :asList) #'<) y)))

;; To test :storeOn we will take a collection, send it :storeOn, eval the returned expression,
;; then compare that the classes and contents are the same. If the collection class isn't an OrderedCollection
;; then we will sort the contents prior to comparing them.

(defun testStoreOn (x)
       (prog ((y (eval (send x :storeOn))))
            (unless (objectp y) (format t "Created collection is not an object") (return nil))
            (unless (eq (send x :class) (send y :class))
                    (format t "Created collection is wrong class - ~s~%" (send y :class))
                    (return nil))
            (unless
             (if (send x :isKindOf OrderedCollection)
                 (equal (send x :asList) (send y :asList))
                 (equal (sort (send x :asList) #'<) (sort (send y :asList) #'<)))
             (format t "Not the same:~%  ~s~%  ~s~%" x y)
             (return nil))
            (return t)))
            

; TESTS GO HERE

;; Notable -- we aren't testing :prin1 or :storeon

;; SET testing
(setq x (send Set :new))
(should-be-true (send x :isEmpty))
(send x :add 3 5)
(should-be-false (send x :isEmpty))
(send x :add 6 0)
(send x :add 1)
(send x :add 3)
(send x :addAll '(1 2 3 4 5))
(should-be-true (cmp x '(1 2 3 4 5)))
(should-be-true (send x :includes 2))
(send x :remove 2)
(should-be-false (send x :includes 2))
(send x :remove 2)
(should-be-true (cmp x '(1 3 4 5)))
(should-be-true (eql (send x :size) 4))
(should-be-true (cmp (send x :select #'evenp) '(4) Set))
(should-be-true (cmp (send x :collect #'1+) '(2 4 5 6) Set))
(should-be-true (eql (let ((cnt 0)) (send x :do (lambda (x) (incf cnt x))) cnt) 13))
(should-fail (send x :addFirst 10))
(should-fail (send x :addLast 10))
(should-fail (send x :removeFirst))
(should-fail (send x :removeLast))
(should-fail (send x :at 1))
(should-fail (send x :atPut 1 0))
(setq y (send Set :new))
(send y :addAll x)
(should-be-true (cmp y '(1 3 4 5)))
(should-be-true (testStoreOn y))
(should-be-true (equal (sort (send (send Set :new :withAll '("A" "B" "A")) :asList) #'string<) '("A" "B")))

;; IDENTITYSET testing
(setq x (send IdentitySet :new))
(should-be-true (send x :isEmpty))
(send x :add 3 5)
(should-be-false (send x :isEmpty))
(send x :add 6 0)
(send x :add 1)
(send x :add 3)
(send x :addAll '(1 2 3 4 5))
(should-be-true (cmp x '(1 2 3 4 5)))
(should-be-true (send x :includes 2))
(send x :remove 2)
(should-be-false (send x :includes 2))
(send x :remove 2)
(should-be-true (cmp x '(1 3 4 5)))
(should-be-true (eql (send x :size) 4))
(should-be-true (cmp (send x :select #'evenp) '(4) IdentitySet))
(should-be-true (cmp (send x :collect #'1+) '(2 4 5 6) IdentitySet))
(should-be-true (eql (let ((cnt 0)) (send x :do (lambda (x) (incf cnt x))) cnt) 13))
(should-fail (send x :addFirst 10))
(should-fail (send x :addLast 10))
(should-fail (send x :removeFirst))
(should-fail (send x :removeLast))
(should-fail (send x :at 1))
(should-fail (send x :atPut 1 0))
(setq y (send IdentitySet :new))
(send y :addAll x)
(should-be-true (cmp y '(1 3 4 5)))
(should-be-true (testStoreOn y))
(should-be-false (equal (sort (send (send IdentitySet :new :withAll '("A" "B" "A")) :asList) #'string<) '("A" "B")))

;; BAG testing 
(setq x (send Bag :new))
(should-be-true (send x :isEmpty))
(send x :add 3)
(should-be-false (send x :isEmpty))
(send x :add 6 0)
(send x :add 1 2)
(send x :add 3)
(send x :addAll '(1 2 3 4 5))
(should-be-true (cmp x '(1 1 1 2 3 3 3 4 5)))
(should-be-true (send x :includes 2))
(send x :remove 2)
(should-be-false (send x :includes 2))
(send x :remove 2)
(send x :remove 3)
(should-be-true (cmp x '(1 1 1 3 3 4 5)))
(should-be-true (eql (send x :size) 7))
(should-be-true (cmp (send x :select #'evenp) '(4) Bag))
(should-be-true (cmp (send x :collect #'1+) '(2 2 2 4 4 5 6)))
(should-be-true (eql (let ((cnt 0)) (send x :do (lambda (x) (incf cnt x))) cnt) 18))
(should-fail (send x :addFirst 10))
(should-fail (send x :addLast 10))
(should-fail (send x :removeFirst))
(should-fail (send x :removeLast))
(should-fail (send x :at 1))
(should-fail (send x :atPut 1 0))
(setq y (send Bag :new))
(send y :addAll x)
(should-be-true (cmp y '(1 1 1 3 3 4 5)))
(should-be-true (testStoreOn y))
(should-be-true (cmp (send y :asSet) '(1 3 4 5) Set))
(should-be-true (cmp (send (send y :asSet) :asBag) '(1 3 4 5) Bag))
(should-be-true (eql (send (send y :asBag) :class) Bag))
(should-be-true (eql (send (send y :asSet) :class) Set)) 

;; DICTIONARY testing
(setq x (send Dictionary :new))
(should-be-true (send x :isEmpty))
(send x :atPut 'a 10)
(should-be-false (send x :isEmpty))
(send x :atPut 'b 20)
(should-work (send x :removeKey 'a :ifAbsent (lambda () (error "fail"))))
(should-work (send x :removeKey 'b :ifAbsent (lambda () (error "fail"))))
(should-fail (send x :removeKey 'c :ifAbsent (lambda () (error "fail"))))
(should-be-true (send x :isEmpty))
(send x :addAll '((a 1)(b 2)(c 3)(d 4)))
(should-be-true (eq (send x :at 'b) 2))
(should-be-true (eq (send x :at 'b :ifAbsent (lambda () 'x)) 2))
(should-be-true (eq (send x :at 'e :ifAbsent (lambda () 'x)) 'x))
(send x :atPut 'd 10)
(should-be-true (cmp x '(1 2 3 10)))
(should-be-false (set-exclusive-or (send (send x :keys) :asList) '(a b c d)))
(should-be-true (cmp (send x :values) '(1 2 3 10) Bag))
(should-be-true (testStoreOn x))
(should-be-true (eql (send x :size) 4))
(should-be-true (send x :includes 3))
(should-be-false (send x :includes 4))
(should-fail (send x :select #'evenp))
(should-fail (send x :collect #'1+))
(should-be-true (eql (let ((cnt 0)) (send x :do (lambda (x) (incf cnt x))) cnt) 16))
(should-be-true (eq (send (send x :keys) :class) Set))
(should-be-true (eq (send (send x :values) :class) Bag))
(should-be-true (equal (send (send (send Dictionary :new :withAll '((1 a) (2 b))) :associations) :asList) '((2 b)(1 a))))
(should-be-true (equal (send (send (send (send OrderedCollection :new :withAll '(a b)) :asDictionary) :associations) :asList) '((1 b)(0 a))))

(should-fail (send x :add 10))
(should-fail (send x :addFirst 10))
(should-fail (send x :addLast 10))
(should-fail (send x :remove 10))
(should-fail (send x :removeFirst))
(should-fail (send x :removeLast))

(setq x (send IdentityDictionary :new))
(should-be-true (send x :isEmpty))
(send x :atPut 'a 10)
(should-be-false (send x :isEmpty))
(send x :atPut 'b 20)
(should-work (send x :removeKey 'a :ifAbsent (lambda () (error "fail"))))
(should-work (send x :removeKey 'b :ifAbsent (lambda () (error "fail"))))
(should-fail (send x :removeKey 'c :ifAbsent (lambda () (error "fail"))))
(should-be-true (send x :isEmpty))
(send x :addAll '((a 1)(b 2)(c 3)(d 4)))
(should-be-true (eq (send x :at 'b) 2))
(should-be-true (eq (send x :at 'b :ifAbsent (lambda () 'x)) 2))
(should-be-true (eq (send x :at 'e :ifAbsent (lambda () 'x)) 'x))
(send x :atPut 'd 10)
(should-be-true (cmp x '(1 2 3 10)))
(should-be-false (set-exclusive-or (send (send x :keys) :asList) '(a b c d)))
(should-be-true (cmp (send x :values) '(1 2 3 10) Bag))
(should-be-true (testStoreOn x))
(should-be-true (eql (send x :size) 4))
(should-be-true (send x :includes 3))
(should-be-false (send x :includes 4))
(should-fail (send x :select #'evenp))
(should-fail (send x :collect #'1+))
(should-be-true (eql (let ((cnt 0)) (send x :do (lambda (x) (incf cnt x))) cnt) 16))
(should-be-true (eq (send (send x :keys) :class) Set))
(should-be-true (eq (send (send x :values) :class) Bag))
(should-be-true (equal (send (send (send IdentityDictionary :new :withAll '((1 a) (2 b))) :associations) :asList) '((2 b)(1 a))))
(should-be-true (equal (send (send (send (send OrderedCollection :new :withAll '(a b)) :asIdentityDictionary) :associations) :asList) '((1 b)(0 a))))

(should-fail (send x :add 10))
(should-fail (send x :addFirst 10))
(should-fail (send x :addLast 10))
(should-fail (send x :remove 10))
(should-fail (send x :removeFirst))
(should-fail (send x :removeLast))

;; ARRAY testing
(setq x (send Bag :new))
(send x :addAll '(11 12 13 14 15))
(setq x (send x :asArray))
(should-be-true (cmp x '(11 12 13 14 15) Array))
(should-be-true (eql (send x :size) 5))
(should-be-false (send x :isEmpty))
(setq x (send Array :new :withAll '(11 12 13 14 15)))
(should-be-true (send x :includes '12))
(should-be-false (send x :includes '20))
(should-be-true (eql (send x :at 1) 12))
(send x :atput 1 22)
(should-be-true (eql (send x :at 1) 22))
(should-be-true (eql (send x :first) 11))
(should-be-true (eql (send x :last) 15))
(setq x (send Array :new :withAll '(30 31 32 33 34)))
(should-be-true (testStoreOn x))
(should-be-true (cmp (send x :select #'evenp) '(30 32 34) Array))
(should-be-true (cmp (send x :collect #'1+) '(31 32 33 34 35) Array))
(should-be-true (eql (let ((cnt 0)) (send x :do (lambda (x) (incf cnt x))) cnt) 160))
(should-be-true (eql (send (send Array :new :size 10) :size) 10))
(should-be-true (cmp (send x :asBag) '(30 31 32 33 34) Bag))
(should-work (send x :addFirst 10))
(should-work (send x :addLast 10))
(should-be-true (equal (send  x :asList) '(10 30 31 32 33 34 10)))
(should-work (send x :removeFirst))
(should-work (send x :removeLast))
(should-be-true (equal (send  x :asList) '(30 31 32 33 34)))
(should-work (send x :add 10))
(should-work (send x :remove 31))
(should-be-true (equal (send  x :asList) '(30 32 33 34 10)))

;; ORDEREDCOLLECTION testing
(setq x (send Array :new :withAll '(11 12 13 14 15)))
(setq x (send x :asOrderedCollection))
(should-be-true (eq (send x :class) OrderedCollection))
(should-be-true (equal (send x :asList) '(11 12 13 14 15)))
(should-be-true (eql (send x :size) 5))
(should-be-false (send x :isEmpty))
(setq x (send OrderedCollection :new))
(should-be-true (send x :isEmpty))
(send x :add 10)
(send x :addLast 20)
(send x :addAll '(30 40))
(send x :addFirst 0)
(should-be-true (eql (send x :at 1) 10))
(should-be-true (eql (send x :size) 5))
(should-be-true (eql (send x :at 0) 0))
(should-be-true (eql (send x :first) 0))
(should-be-true (eql (send x :last) 40))
(should-be-true (send x :includes '10))
(should-be-false (send x :includes '15))
(send x :atPut 2 23)
(should-be-true (eql (send x :at 2) 23))
(should-be-true (cmp x '(0 10 23 30 40) OrderedCollection))
(should-be-true (cmp (send x :select #'evenp) '(0 10 30 40) OrderedCollection))
(should-be-true (cmp (send x :collect #'1+) '(1 11 24 31 41) OrderedCollection))
(should-be-true (eql (let ((cnt 0)) (send x :do (lambda (x) (incf cnt x))) cnt) 103))
(send x :removeFirst)
(should-be-true (equal (send x :asList) '(10 23 30 40)))
(should-be-true (testStoreOn x))
(send x :removeLast)
(should-be-true (equal (send x :asList) '(10 23 30)))
(should-be-true (eql (send x :at 0) 10))
(should-be-true (eql (send x :size) 3))
(should-be-true (send x :remove 30))
(should-be-true (cmp (send x :asArray) '(10 23) Array))

;; SORTEDCOLLECTION testing
(setq x (send Array :new :withAll '(1 5 3 2 4)))
(setq x (send x :asSortedCollection))
(should-be-true (eq (send x :class) SortedCollection))
(should-be-true (equal (send x :asList) '(1 2 3 4 5)))
(should-be-true (eql (send x :size) 5))
(should-be-false (send x :isEmpty))
(setq x (send SortedCollection :new))
(should-be-true (send x :isEmpty))
(send x :add 10)
(send x :add 5)
(send x :addAll '(30 40))
(send x :add 20)
(should-be-true (eql (send x :at 2) 20))
(should-be-true (eql (send x :size) 5))
(should-be-true (eql (send x :at 0) 5))
(should-be-true (eql (send x :first) 5))
(should-be-true (eql (send x :last) 40))
(should-be-true (send x :includes '20))
(should-be-false (send x :includes '50))
(should-fail (send x :atPut 1 23))
(should-be-true (cmp x '(5 10 20 30 40) SortedCollection))
(should-be-true (cmp (send x :select #'evenp) '(10 20 30 40) SortedCollection))
(should-be-true (cmp (send x :collect #'1+) '(6 11 21 31 41) SortedCollection))
(should-be-true (equal (send (send (send x :asSortedCollection :cmpfcn #'>) :select #'evenp)  :asList)
                       '(40 30 20 10)))
(should-be-true (equal (send (send (send x :asSortedCollection :cmpfcn #'>) :collect #'1+) :asList)
                       '(41 31 21 11 6)))
(should-be-true (eql (let ((cnt 0)) (send x :do (lambda (x) (incf cnt x))) cnt) 105))
(should-be-true (equal (send (send x :asSortedCollection :cmpfcn #'>) :asList) '(40 30 20 10 5)))
(send x :removeFirst)
(should-be-true (equal (send x :asList) '(10 20 30 40)))
(should-be-true (testStoreOn x))
(should-be-true (testStoreOn (send SortedCollection :new :cmpFcn #'> :withAll '(1 2 3 4 5 6))))
(send x :removeLast)
(should-be-true (equal (send x :asList) '(10 20 30)))
(should-be-true (eql (send x :at 0) 10))
(should-be-true (eql (send x :size) 3))
(should-be-true (send x :remove 30))
(should-be-true (cmp x '(10 20)))
(should-fail (send x :addFirst 0))
(should-fail (send x :addLast 100))

;; Interval testing
(setq x (send Interval :new 2 10 2))
(should-be-true (testStoreOn x))
(should-be-true (eql (send x :at 3) 8))
(should-be-false (send x :isEmpty))
(should-be-true (eql (send x :size) 5))
;; (should-fail (send x :at 5))  We aren't doing range checks
(should-be-true (send x :includes 10))
(should-be-false (send x :includes 11))
(should-be-true (equal (send x :asList) '(2 4 6 8 10)))
(should-be-true (equal (send (send Interval :new 10 1 -1) :asList) '(10 9 8 7 6 5 4 3 2 1)))
(should-be-true (equal (send (send x :asSortedCollection :cmpfcn #'>) :asList) '(10 8 6 4 2)))
(setq x (send Interval :new 1 10))
(should-be-true (cmp (send x :select #'evenp) '(2 4 6 8 10) OrderedCollection))
(should-be-true (cmp (send x :collect #'1+) '(2 3 4 5 6 7 8 9 10 11) OrderedCollection))
(should-be-true (eql (let ((cnt 0)) (send x :do (lambda (x) (incf cnt x))) cnt) 55))
(should-fail (send x :addFirst 10))
(should-fail (send x :addLast 10))
(should-fail (send x :removeFirst))
(should-fail (send x :removeLast))
(should-fail (send x :add 10))
(should-fail (send x :remove 1))
(should-fail (send x :atPut 2 10))



(terpri)

(princ test-number)
(princ " tests performed.")
(terpri)

(princ bugs-found)
(princ " bugs found.")
(terpri)
