(require "factor")
(unless (find-package "EULER")
        (make-package "EULER" :use '("XLISP" "FACTOR")))
(use-package "EULER")
(in-package "EULER")
(export '(euler31 euler32 euler33 euler34 euler35 euler35a euler36
                  euler37 euler38 euler39 euler40))
(require "fact")

(Defun Euler31 (Money &Optional (Coins '(200 100 50 20 10 5 2 1)))
       (If (Null (Cdr Coins))
	   1	; Only One Choice If Only One Coin Type Left
	   (Let ((Ways 0))
		(Do ((Quant 0 (+ (Car Coins) Quant))) ; Take Varying Amounts Of First Coin In List
		    ((> Quant Money)) ; Stop When Too Many Coins
		    (Incf Ways (Euler31 (- Money Quant) (Cdr Coins))))
		Ways)))

; Values can be 1 digit times 4 digit with a 4 digit product, in which case the 4 digit
; value would be at least 1234 but less than 4987 and the single digit must be greater than 1,
; or a 2 digit times 3 digit where the three digit number is at least 123 but less than or
; equal to 498 and the two digit number at least 12. We can also end the inner loop when the
; product exceeds 4 digits.

(defun euler32 ()
       (let (answer)
	    (do ((i 2 (1+ i)))
		((> i 9))
		(do* ((j 1234 (1+ j)) (prod (* i j)(* i j)))
		    ((> prod 9999))
		    (when
		     (equal "123456789"
			    (sort (format nil "~s~s~s" i j prod) #'char<))
;		     (format t "~s x ~s = ~s\n" i j prod)
		     (setq answer (adjoin prod answer)))))
	    (do ((i 12 (1+ i)))
		((> i 98))
		(do* ((j 123 (1+ j)) (prod (* i j) (* i j)))
		    ((> prod 9999))
		    (when
		     (equal "123456789"
			    (sort (format nil "~s~s~s" i j prod) #'char<))
;		     (format t "~s x ~s = ~s\n" i j prod)
		     (setf answer (adjoin prod answer)))))
	    (apply #'+ answer)))

"My XLISP-PLUS solution, which runs in 73 milliseconds. Unlike the
other LISP solutions I saw here, I do the pandigital check with just a
string. Also XLISP-PLUS doesn't (yet) have the LOOP macro, but I don't
think my solution suffers for it's absence."

(defun euler33 ()
       (let ((result 1))
       (do ((j 1 (1+ j)))
	   ((> j 9))
	   (do ((i 1 (1+ i)))
	       ((>= i j))
	       (do ((k i (1+ k)))
		   ((> k 9))
		   (let ((ij (+ (* i 10) j))
			 (jk (+ (* j 10) k)))
			(when (eql (* ij k) (* jk i))
			      (format t "~s ~s \n" ij jk)
			      (setf result (* result (/ i k))))))))
       result))

"XLISP-PLUS, 420 microseconds.  Brute force with reasonable limits set
for each digit. Simple nested loops but relies on LISP's rational
    arithmetic to automatically reduce the resulting fractions."

(defun euler34 ()
       (let ((result 0)
	     (factarray  (let ((result (list 1)))
			      (do ((i 1 (+ 1 i)))
				  ((> i 9))
				  (push (facti i) result))
			      (coerce (nreverse result) 'array))))
	    (do ((i 3 (+ i 1)))
		((> i 2540160))
		(when (eql i
			   (apply #'+ (map 'list
					   #'(lambda (x) (aref factarray (digit-char-p x)))
					   (princ-to-string i))))
		      (format t "~s\n" i)
		      (incf result i)))
	    result))

(labels (
         (is-circular-prime (n)
                            (let* ((len (length (format nil "~s" n)))
                                   (dig (expt 10 (1- len))))
                                  (dotimes (i (1- len))
                                           (setf n (let ((v (multiple-value-list (floor n dig))))
                                                        (+ (car v) (* 10 (cadr v)))))
                                           (unless (primep n)
                                                   (return-from is-circular-prime nil)))
                                  t)))

        (defun euler35 ()
               (let ((count 0))
                    (do ((i 2 (nextprime i)))
                        ((> i 1000000))
                        (when (is-circular-prime i)
                              (format t "~s\n" i)
                              (incf count)))
                    count)))

"XLISP-PLUS. I went through a list of all primes < 1,000,000, rotated
them and checked for primality. The functions I used for primes only
built a table up to the square-root of the number being tested, which
isn't the best way to go for this problem. Someday I may rewrite it
(yeah, sure).  "

;; Alternative solution using array of primes
(defvar *notprimes*)

(labels (
         (sieve (n &aux (limit (/ n 2)))
                (setf *notprimes* (make-array n))
                (setf (aref *notprimes* 0) t)
                (setf (aref *notprimes* 1) t)
                (do ((i 2 (1+ i)))
                    ((> i limit))
                    (when (null (aref *notprimes* i))
                          (do ((j (* 2 i) (+ j i)))
                              ((>= j n))
                              (setf (aref *notprimes* j) t)))))

         (is-circular-prime-a (n)
                              (let* ((len (length (format nil "~s" n)))
                                     (dig (expt 10 (1- len))))
                                    (dotimes (i (1- len))
                                             (setf n (let ((v (multiple-value-list (floor n dig))))
                                                          (+ (car v) (* 10 (cadr v)))))
                                             (when (aref *notprimes* n)
                                                   (return-from is-circular-prime-a nil)))
                                    t)))

        (defun euler35a ()
               (sieve 999999)
               (let ((count 1)) ; start with value 2 already "found"
                    (do ((i 3 (+ i 2)))
                        ((> i 999998))
                        (when (and (null (aref *notprimes* i))
                                   (is-circular-prime-a i))
                              (format t "~s\n" i)
                              (incf count)))
                    count)))

"OK, I rewrote it using a sieve to find the primes in advance. The
prime number package I had previously written was optimized for
finding the factors of individual very large numbers as opposed to
finding the primeness of a large quantity of numbers. Time went from
11.65 seconds to 4.52 (for sieve) plus 1.53 seconds for the problem."

(defun euler36 ()
       (let ((answer 0))
	    (do* ((i 1 (+ i 2))
		  (str10 "1" (format nil "~s" i)))
		 ((> i 999999))
		 (when (equal str10 (reverse str10))
		       (let ((str2 (format nil "~2r" i)))
			    (when (equal str2 (reverse str2))
				  (format t "~a ~a\n" str10 str2)
				  (incf answer i)))))
	    answer))

"We know the values have to be odd."

(defun  euler37 ()
	(let ((result 0) (count 0))
	     (do ((failflag nil nil)
		  (i (nextprime 7) (nextprime i)))
		 ((eql count 11) result)
		 (do ((left (mod i (expt 10 (floor (log i 10))))
			    (mod left (expt 10 (floor (log left 10)))))
		      (right (floor i 10) (floor right 10)))
		     ((or failflag (zerop right)))
		     (unless (and (primep left) (primep right))
			     (setf failflag t)))
		 (unless failflag
			 (format t "~s\n" i)
			 (incf result i)
			 (incf count)))
	     result))

"Brute force. No upper limit needed to be set since problem statement said there were 11 values! "

(labels (
         (sieve (n)
                (setf *notprimes* (make-array n))
                (setf (aref *notprimes* 0) t)
                (setf (aref *notprimes* 1) t)
                (do ((i 2 (1+ i)))
                    ((> i (/ n 2)))
                    (when (null (aref *notprimes* i))
                          (do ((j (* 2 i) (+ j i)))
                              ((>= j n))
                              (setf (aref *notprimes* j) t))))))

        (defun  euler37a ()
                (sieve 999999)
                (let ((result 0) (count 0))
                     (do ((failflag nil nil)
                          (i 11 (+ i 2)))
                         ((eql count 11) result)
                         (unless (aref *notprimes* i)
                                 (do ((left (mod i (expt 10 (floor (log i 10))))
                                            (mod left (expt 10 (floor (log left 10)))))
                                      (right (floor i 10) (floor right 10)))
                                     ((or failflag (zerop right)))
                                     (when (or (aref *notprimes* left)
                                               (aref *notprimes* right))
                                           (setf failflag t)))
                                 (unless failflag
                                         (format t "~s\n" i)
                                         (incf result i)
                                         (incf count))))
                     result)))

"Then I rewrote using a sieve. Time was reduced from 8.49 to .84 seconds, but doing the sieve takes an additional 4.5.
Additionally, this only works because I know the 11th value is less than a million."


; By inspection we know that the sequence length must be 2. Reasoning:
; 1. Number must start with 9, otherwise 918273645 given in problem would be best.
; 2. For a sequence of 3 98 gives 8 digits and 921 gives 11 digits.
; 3. There are no possible values for a sequence of 4, 5 is shown, and 6 or more would
;    likewise be impossible
; 4. For n=2 the answer must be in the range of 9123 through 9876. We start at the top
;    and go down. There had better be a solution!

(defun euler38 ()
       (let ((result nil))
	    (do ((i 9876 (1- i)))
		(result result)
		(let ((string (format nil "~s~s" i (* 2 i))))
		     (when (equal (sort string #'char<)
				  "123456789")
			   ; we know we have 4 digit in front of 5 digit to get 9
			   (setf result (+ (* i 100000) (* 2 i))))))))
 

"1.97 milliseconds"

(defun euler39 ()
       (let ((bestp 0) (lenbestp 0))
	    (do ((len 0 0)
		 (p 502 (+ p 2)))
		((> p 1000))
		(do ((a (ceiling (/ p 3.14159)) (1- a)))
		    ((< a 2))
		    (when (zerop (rem (* p (- p (* a 2))) (* 2 (- p a))))
			  (incf len)
			  ))
		(when (> len lenbestp)
		      (setf bestp p lenbestp len))
		)
	    (list bestp lenbestp)))

"Knowing a*a + b*b = c*c and a+b+c=p, solved for b=p(p-2a)/2(p-a). By
inspection we know that P is even and we only have to check 502
through 1000 inclusive. For each p I check out values of a > 1 and <=
ceiling(p/(2+sqrt(2))). 67msec in XLISP-PLUS.  " 


(defun euler40 ()
       (let ((result 1) (terms (list 1 10 100 1000 10000 100000 1000000)))
	    (do* ((i 1 (1+ i))
		  (pos 1 (incf pos (length (format nil "~s" i)))))
		 ((null terms) result)
		 (when (>= pos (car terms))
		       (let* ((val (format nil "~s" i))
			      (digit (digit-char-p (aref val (- (car terms)
								(- pos (length val))
								1)))))
;			     (format t "~s\n" digit)
			     (setf result (* digit result))
			     (setf terms (cdr terms)))))))

"My first attempt involved creating a 1,000,000 character long list
(XLISP-PLUS wasn't compiled to handle a 1,000,000 character long
string and I didn't want to recompile.) Unfortunately it took too long
to run. So I rewrote the solution to create the running product while
generating the million digits, thus eliminating the need to save them.
0.23 seconds."

(in-package "USER")
