(require "factor")
(unless (find-package "EULER")
        (make-package "EULER" :use '("XLISP" "FACTOR")))
(use-package "EULER")
(in-package "EULER")
(export '(euler51 euler52 euler53 euler53a euler54
		  euler55 euler56
                  euler57 euler58 euler59 euler60))
(require "fact")

(defun euler51 ()
       (flet ((check8 (val dig)
		      (let ((count 0)
			    (lval (coerce val 'list)))
			   (mapc #'(lambda (d)
					   (let ((v (read-from-string (coerce (subst d dig lval)
									      'string))))
						(when (and (>= v 100000) (primep v))
						      (incf count))))
				 '(#\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9))
			   (when (eql count 8)
				 (format t "~s" val)
				 t))))

	     (do* ((i (nextprime 100000) (nextprime i))
		   (found nil)
		   (istr (format nil "~s" i) (format nil "~s" i)))
		  ((or (> i 999999)
		       (cond ((eql 3 (count #\0 istr))(check8 istr #\0))
			     ((eql 3 (count #\1 istr))(check8 istr #\1))
			     ((eql 3 (count #\2 istr))(check8 istr #\2))
			     (t nil)))))))

; Not that difficult but for some reason I made lots of syntax errors.
; Straightforward -- generate the next prime, check for triplicate digit
; 0, 1, or 2 (since one of these must be in solution), then check that
; value substituting 1, 2, 3, ... 9 looking for 8 primes.

(defun euler52 ()
       (do* ((i 100000 (1+ i))
	     (istr (sort (princ-to-string i) #'char<)
		   (sort (princ-to-string i) #'char<)))
	    ((> i 170000))
	    (when (and (equal istr (sort (princ-to-string (* 2 i)) #'char<))
		       (equal istr (sort (princ-to-string (* 3 i)) #'char<))
		       (equal istr (sort (princ-to-string (* 4 i)) #'char<))
		       (equal istr (sort (princ-to-string (* 5 i)) #'char<))
		       (equal istr (sort (princ-to-string (* 6 i)) #'char<)))
		  (format t "~s" i)
		  (return-from euler52))))

; Brute force in XLISP-PLUS. Very clean looking and very fast (180 msec).

(defun euler53 (&aux (result 0))
       (do ((n 1 (1+ n)))
	   ((> n 100))
	   (do ((r 1 (1+ r)))
	       ((eql n r))
	       (let ((nCr (floor (facti n) (* (facti r) (facti (- n r))))))
		    (when (> ncr 1000000)
;			  (format t "~s ~s\n" n r)
			  (incf result)))))
       result)

; I could have used symmetry, simplify the calculation of n!/r!, or use
; memoization. But it ran so fast with brute force I just used that.
; XLISP-PLUS 0.39 seconds.

(defun euler53a (&aux (result 0) (facts (make-array 101)))
       (setf (aref facts 0) 0)
       (setf (aref facts 1) 1)
       (do* ((i 2 (1+ i))
	     (facti i (* i facti)))
	    ((> i 100))
	    (setf (aref facts i) facti))
       (do ((n 1 (1+ n)))
	   ((> n 100))
	   (do ((r 1 (1+ r)))
	       ((eql n r))
	       (let ((nCr (floor (aref facts n) (* (aref facts r) (aref facts (- n r))))))
		    (when (> ncr 1000000)
;			  (format t "~s ~s\n" n r)
			  (incf result)))))
       result)

(defun euler54 (&aux spots)
       (labels
	((parse-hand (str)
		     (list (gethash (aref str 0) spots)
			   (gethash (aref str 3) spots)
			   (gethash (aref str 6) spots)
			   (gethash (aref str 9) spots)
			   (gethash (aref str 12) spots)
			   (let ((flushed (coerce (list
						   (aref str 1)
						   (aref str 4)
						   (aref str 7)
						   (aref str 10)
						   (aref str 13))
						  'string)))
				(or (equal flushed "SSSSS")
				    (equal flushed "HHHHH")
				    (equal flushed "DDDDD")
				    (equal flushed "CCCCC")))))

	 (evalhand (hand)
		   ; We will rate hands:
		   ; 0        high card
		   ;   and (((c1*13+c2)*13+c3)*13+c4)*13+c5 to differentiate
		   ; 1000000  one pair
		   ;   and ((pair*13+c2)*13+c3)*13+c4 to differentiate
		   ; 2000000  two pairs
		   ;   and (hpair*13+lpair)*13+c5 to differentiate
		   ; 3000000  three of a kind
		   ;   and three to differentiate
		   ; 4000000  straight
		   ;   and topcard to differentiate
		   ; 5000000  flush
		   ;   and (((c1*13+c2)*13+c3)*13+c4)*13+c5 to differentiate
		   ; 6000000  full house
		   ;   and three to differentiate
		   ; 7000000  four of a kind
		   ;   and four to differentiate
		   ; 8000000  straight/royal flush
		   ;   and topcard to differentiate
		   (let ((flushed (sixth hand))
			 (sorted (sort (cdr (reverse hand)) #'>)))
			(cond ((or (eql (first sorted) (fourth sorted)) ; 4 of a kind
				   (eql (second sorted) (fifth sorted)))
			       (+ 7000000 (second sorted)))
			      ((or (and (eql (first sorted) (third sorted)) ; full house
					(eql (fourth sorted) (fifth sorted)))
				   (and (eql (first sorted) (second sorted))
					(eql (third sorted) (fifth sorted))))
			       (+ 6000000 (third sorted)))
			      ((and (eql (first sorted) (1+ (second sorted))) ; straight
				    (eql (second sorted) (1+ (third sorted)))
				    (eql (third sorted) (1+ (fourth sorted)))
				    (eql (fourth sorted) (1+ (fifth sorted))))
			       (if flushed ; allow for straight flush
				   (+ 8000000 (first sorted))
				   (+ 4000000 (first sorted))))
			      (flushed ; flush (other than straight flush)
				       (+ (reduce #'(lambda (x y) (+ (* x 13) y)) sorted) 5000000))
			      ((or (eql (first sorted) (third sorted)) ; three of a kind
				   (eql (second sorted) (fourth sorted))
				   (eql (third sorted) (fifth sorted)))
			       (+ 3000000 (third sorted)))
			      ((or (and (eql (first sorted) (second sorted)) ; Two pair
					(or (eql (third sorted) (fourth sorted))
					    (eql (fourth sorted) (fifth sorted))))
				   (and (eql (second sorted) (third sorted))
					(eql (fourth sorted) (fifth sorted))))
			       (+ 2000000 (* 13 (second sorted)) (fourth sorted)))
			      ((eql (first sorted) (second sorted)) ; one pair, first possibility
			       (+ (reduce #'(lambda (x y) (+ (* x 13) y)) (cdr sorted)) 1000000))
			      ((eql (second sorted) (third sorted)) ; one pair, second possibility
			       (+ (reduce #'(lambda (x y) (+ (* x 13) y))
					  (cons (second sorted)
						(cons (first sorted) (cdddr sorted))))
				  1000000))
			      ((eql (third sorted) (fourth sorted)) ; one pair, third possibility
			       (+ (reduce #'(lambda (x y) (+ (* x 13) y))
					  (list (third sorted)
						(first sorted)
						(second sorted)
						(fifth sorted)))
				  1000000))
			      ((eql (fourth sorted) (fifth sorted)) ; one pair, fourth possibility
			       (+ (reduce #'(lambda (x y) (+ (* x 13) y))
					  (list (fourth sorted)
						(first sorted)
						(second sorted)
						(third sorted)))
				  1000000))
			      (t ; high card
				 (reduce #'(lambda (x y) (+ (* x 13) y)) sorted)))))

	 (readhands ()
		    (let ((fp (open "poker.txt")) results)
			 (do ((line (read-line fp nil) (read-line fp nil)))
			     ((null line))
			     (push (list (parse-hand line)
					 (parse-hand (subseq line 15)))
				   results))
			 (close fp)
			 results)))

	(setf spots (make-hash-table :test #'eq))
	(setf (gethash #\2 spots) 0)
	(setf (gethash #\3 spots) 1)
	(setf (gethash #\4 spots) 2)
	(setf (gethash #\5 spots) 3)
	(setf (gethash #\6 spots) 4)
	(setf (gethash #\7 spots) 5)
	(setf (gethash #\8 spots) 6)
	(setf (gethash #\9 spots) 7)
	(setf (gethash #\T spots) 8)
	(setf (gethash #\J spots) 9)
	(setf (gethash #\Q spots) 10)
	(setf (gethash #\K spots) 11)
	(setf (gethash #\A spots) 12)

	(reduce #'+ (mapcar #'(lambda (x) (if (> (evalhand (first x)) (evalhand (second x)))
					      1
					      0)) 
			    (readhands)))))

; 27 milliseconds in XLISP-PLUS. I thought this a very straightforward
; problem. I read the file and parsed into hands of numeric card values
; and flushness. I assigned a score to each hand, checking in order
; 4-of-a-kind, full house, straight (including straight flush), flush,
; three-of-a-kind, two pair, pair, and high card.

; It took longer to write than any others I've done so far, but was one of the most enjoyable.


(defun euler55 ()
       (flet ((reversed (val)
			(read-from-string (nreverse (princ-to-string val))))

	      (palindromep (val)
			   (let ((str (princ-to-string val)))
				(equal str (reverse str)))))

	     (let ((result 0))
		  (do ((i 1 (1+ i)))
		      ((eql i 10000))
		      (do ((iter 1 (1+ iter))
			   (val (+ i (reversed i)) (+ val (reversed val))))
			  ((or (palindromep val) (eql iter 50))
			   (when (eql iter 50)
 ;			         (format t "~s\n" i)
				 (incf result)
				 t))))
		  result)))

; 286 milliseconds in XLISP-PLUS. Straight forward use of writing and reading from strings and the REVERSE function.


(defun euler56 ()
       (let ((result 0))
	    (do ((a 2 (1+ a)))
		((> a 99))
		(do ((b 2 (1+ b)))
		    ((> b 99))
		    (let* ((val (expt a b))
			   (sum (reduce #'+
					(map 'array
					     #'digit-char-p
					     (princ-to-string val)))))
			  (when (> sum result)
;				(format t "~s ~s ~s\n" a b sum)
				(setf result sum)))))
	    result))
 
; No assumptions made other than ignoring a,b < 2. 140 msec in XLISP-PLUS.


 (defun euler57 ()
       (let ((result 0))
	    (do ((iter 1 (1+ iter))
		 (val 3/2 (+ 1 (/ 1 (+ 1 val)))))
		((> iter 1000))
		(when (> (length (princ-to-string (numerator val)))
			 (length (princ-to-string (denominator val))))
		      (incf result)))
	    result))

; XLISP-PLUS, 1.56 seconds. Brute force using the built-in rational arithmetic.

(defun euler58 ()
 (flet ((primepn (val) ; We want a primep that returns a numeric value rather than T or NIL
		 (if (primep val) 1 0)))

       (do* ((primes 0 (+ primes
			  (primepn (+ dist n))
			  (primepn (+ dist (* n 2)))
			  (primepn (+ dist (* n 3))))) ; At n*4 it will never be prime
	     (dist 1 (+ dist (* n 4))) ; Distance to start of new circle
	     (n 2 (+ n 2)) ; length of side
	     (ave 1.0 (/ primes (* n 2)))) ; Average set to 1 so we don't leave early
	    ((< ave 0.1)
	     (1+ n)))))

; Based on knowledge from earlier problem. XLISP-PLUS, 6.5 seconds.

(defconstant +text+ #(79 59 12 2 79 35 8 28 20 2 3 68 8 9 68 45 0 12 9
67 68 4 7 5 23 27 1 21 79 85 78 79 85 71 38 10 71 27 12 2 79 6 2 8 13
9 1 13 9 8 68 19 7 1 71 56 11 21 11 68 6 3 22 2 14 0 30 79 1 31 6 23
19 10 0 73 79 44 2 79 19 6 28 68 16 6 16 15 79 35 8 11 72 71 14 10 3
79 12 2 79 19 6 28 68 32 0 0 73 79 86 71 39 1 71 24 5 20 79 13 9 79 16
15 10 68 5 10 3 14 1 10 14 1 3 71 24 13 19 7 68 32 0 0 73 79 87 71 39
1 71 12 22 2 14 16 2 11 68 2 25 1 21 22 16 15 6 10 0 79 16 15 10 22 2
79 13 20 65 68 41 0 16 15 6 10 0 79 1 31 6 23 19 28 68 19 7 5 19 79 12
2 79 0 14 11 10 64 27 68 10 14 15 2 65 68 83 79 40 14 9 1 71 6 16 20
10 8 1 79 19 6 28 68 14 1 68 15 6 9 75 79 5 9 11 68 19 7 13 20 79 8 14
9 1 71 8 13 17 10 23 71 3 13 0 7 16 71 27 11 71 10 18 2 29 29 8 1 1 73
79 81 71 59 12 2 79 8 14 8 12 19 79 23 15 6 10 2 28 68 19 7 22 8 26 3
15 79 16 15 10 68 3 14 22 12 1 1 20 28 72 71 14 10 3 79 16 15 10 68 3
14 22 12 1 1 20 28 68 4 14 10 71 1 1 17 10 22 71 10 28 19 6 10 0 26 13
20 7 68 14 27 74 71 89 68 32 0 0 71 28 1 9 27 68 45 0 12 9 79 16 15 10
68 37 14 20 19 6 23 19 79 83 71 27 11 71 27 1 11 3 68 2 25 1 21 22 11
9 10 68 6 13 11 18 27 68 19 7 1 71 3 13 0 7 16 71 28 11 71 27 12 6 27
68 2 25 1 21 22 11 9 10 68 10 6 3 15 27 68 5 10 8 14 10 18 2 79 6 2 12
5 18 28 1 71 0 2 71 7 13 20 79 16 2 28 16 14 2 11 9 22 74 71 87 68 45
0 12 9 79 12 14 2 23 2 3 2 71 24 5 20 79 10 8 27 68 19 7 1 71 3 13 0 7
16 92 79 12 2 79 19 6 28 68 8 1 8 30 79 5 71 24 13 19 1 1 20 28 68 19
0 68 19 7 1 71 3 13 0 7 16 73 79 93 71 59 12 2 79 11 9 10 68 16 7 11
71 6 23 71 27 12 2 79 16 21 26 1 71 3 13 0 7 16 75 79 19 15 0 68 0 6
18 2 28 68 11 6 3 15 27 68 19 0 68 2 25 1 21 22 11 9 10 72 71 24 5 20
79 3 8 6 10 0 79 16 8 79 7 8 2 1 71 6 10 19 0 68 19 7 1 71 24 11 21 3
0 73 79 85 87 79 38 18 27 68 6 3 16 15 0 17 0 7 68 19 7 1 71 24 11 21
3 0 71 24 5 20 79 9 6 11 1 71 27 12 21 0 17 0 7 68 15 6 9 75 79 16 15
10 68 16 0 22 11 11 68 3 6 0 9 72 16 71 29 1 4 0 3 9 6 30 2 79 12 14 2
68 16 7 1 9 79 12 2 79 7 6 2 1 73 79 85 86 79 33 17 10 10 71 6 10 71 7
13 20 79 11 16 1 68 11 14 10 3 79 5 9 11 68 6 2 11 9 8 68 15 6 23 71 0
19 9 79 20 2 0 20 11 10 72 71 7 1 71 24 5 20 79 10 8 27 68 6 12 7 2 31
16 2 11 74 71 94 86 71 45 17 19 79 16 8 79 5 11 3 68 16 7 11 71 13 1
11 6 1 17 10 0 71 7 13 10 79 5 9 11 68 6 12 7 2 31 16 2 11 68 15 6 9
75 79 12 2 79 3 6 25 1 71 27 12 2 79 22 14 8 12 19 79 16 8 79 6 2 12
11 10 10 68 4 7 13 11 11 22 2 1 68 8 9 68 32 0 0 73 79 85 84 79 48 15
10 29 71 14 22 2 79 22 2 13 11 21 1 69 71 59 12 14 28 68 14 28 68 9 0
16 71 14 68 23 7 29 20 6 7 6 3 68 5 6 22 19 7 68 21 10 23 18 3 16 14 1
3 71 9 22 8 2 68 15 26 9 6 1 68 23 14 23 20 6 11 9 79 11 21 79 20 11
14 10 75 79 16 15 6 23 71 29 1 5 6 22 19 7 68 4 0 9 2 28 68 1 29 11 10
79 35 8 11 74 86 91 68 52 0 68 19 7 1 71 56 11 21 11 68 5 10 7 6 2 1
71 7 17 10 14 10 71 14 10 3 79 8 14 25 1 3 79 12 2 29 1 71 0 10 71 10
5 21 27 12 71 14 9 8 1 3 71 26 23 73 79 44 2 79 19 6 28 68 1 26 8 11
79 11 1 79 17 9 9 5 14 3 13 9 8 68 11 0 18 2 79 5 9 11 68 1 14 13 19 7
2 18 3 10 2 28 23 73 79 37 9 11 68 16 10 68 15 14 18 2 79 23 2 10 10
71 7 13 20 79 3 11 0 22 30 67 68 19 7 1 71 8 8 8 29 29 71 0 2 71 27 12
2 79 11 9 3 29 71 60 11 9 79 11 1 79 16 15 10 68 33 14 16 15 10 22 73))

(defun euler59 ()
       (flet ((successcalc (key &aux (result 0))
			   (dotimes (i (length +text+))
				    (let ((ch (logxor (aref key (mod i 3)) (aref +text+ i))))
					 (write-char (int-char ch))
					 (incf result ch)))
			   (format t
				   "Key=~a~a~a"
				   (int-char (aref key 0))
				   (int-char (aref key 1))
				   (int-char (aref key 2)))
			   result)

	      (isinvalid (ch) ; Not a valid alpha or space character
			 (or (< ch 32) (> ch 126))))

	     (let ((key #(0 0 0)) (lenm1 (1- (length +text+))))
		  (dotimes
		   (a 26)
		   (setf (aref key 0) (+ a 97))
		   (dotimes
		    (b 26)
		    (setf (aref key 1) (+ b 97))
		    (dotimes
		     (c 26)
		     (setf (aref key 2) (+ c 97))
		     (do* ((i 0 (1+ i))
			   (ch (logxor (aref key 0) (aref +text+ 0))
			       (logxor (aref key (mod i 3)) (aref +text+ i)))
			   (last5 (list 0 0 0 0 0) (nconc (cdr last5) (list ch)))
			   (marker nil (or marker (and (equal last5 '(32 116  104 101 32)))))
			   (fail (isinvalid ch)
				 (isinvalid ch)))
			  ((or fail (eql i lenm1))
			   (unless (or fail (null marker)) ; success!
				   (return-from euler59 (successcalc key)))))))))))

; Brute force, checking for valid plain text characters and the string
; " the ". Execution time is 1.73 seconds, including printing the
; plaintext.

(defvar *pr* nil)
(defvar *prsize* nil)

(defun euler60 (&optional (nmax 10000) &aux (result nil))
       (flet
	((buildprimes (nmax)
		(setf *pr* nil *prsize* nil)
		(do ((i 3 (nextprime i)))
		    ((> i nmax))
		    (push i *pr*)
		    (push (expt 10 (1+ (floor (log i 10)))) *prsize*))
		(setf *pr* (nreverse (coerce *pr* 'array)))
		(setf *prsize* (nreverse (coerce *prsize* 'array)))
		nil))

	(buildprimes nmax)
	(do ((i1 0 (1+ i1)))
	    ((eql i1 (- (length *pr*) 4)))
	    (do ((p1 (aref *pr* i1))
		 (p1s (aref *prsize* i1))
		 (i2 (1+ i1) (1+ i2)))
		((eql i2 (- (length *pr*) 3)))
		(when (and (eql (mod p1 3) (mod (aref *pr* i2) 3))
			   (primep (+ (* p1 (aref *prsize* i2)) (aref *pr* i2)))
			   (primep (+ (* (aref *pr* i2) p1s) p1)))
		      (do ((p2 (aref *pr* i2))
			   (p2s (aref *prsize* i2))
			   (i3 (1+ i2) (1+ i3)))
			  ((eql i3 (- (length *pr*) 2)))
			  (when (and (eql (mod p1 3) (mod (aref *pr* i3) 3))
				     (primep (+ (* p1 (aref *prsize* i3))
						(aref *pr* i3)))
				     (primep (+ (* (aref *pr* i3) p1s) p1))
				     (primep (+ (* p2 (aref *prsize* i3))
						(aref *pr* i3)))
				     (primep (+ (* (aref *pr* i3) p2s) p2)))
				(do ((p3 (aref *pr* i3))
				     (p3s (aref *prsize* i3))
				     (i4 (1+ i3) (1+ i4)))
				    ((eql i4 (1- (length *pr*))))
				    (when (and (eql (mod p1 3) (mod (aref *pr* i4) 3))
					       (primep (+ (* p1 (aref *prsize* i4))
							  (aref *pr* i4)))
					       (primep (+ (* (aref *pr* i4) p1s) p1))
					       (primep (+ (* p2 (aref *prsize* i4))
							  (aref *pr* i4)))
					       (primep (+ (* (aref *pr* i4) p2s) p2))
					       (primep (+ (* p3 (aref *prsize* i4))
							  (aref *pr* i4)))
					       (primep (+ (* (aref *pr* i4) p3s) p3)))
					  (do ((p4 (aref *pr* i4))
					       (p4s (aref *prsize* i4))
					       (i5 (1+ i4) (1+ i5)))
					      ((eql i5 (length *pr*)))
					      (when (and (eql (mod p1 3) (mod (aref *pr* i5) 3))
							 (primep (+ (* p1 (aref *prsize* i5))
								    (aref *pr* i5)))
							 (primep (+ (* (aref *pr* i5) p1s) p1))
							 (primep (+ (* p2 (aref *prsize* i5))
								    (aref *pr* i5)))
							 (primep (+ (* (aref *pr* i5) p2s) p2))
							 (primep (+ (* p3 (aref *prsize* i5))
								    (aref *pr* i5)))
							 (primep (+ (* (aref *pr* i5) p3s) p3))
							 (primep (+ (* p4 (aref *prsize* i5))
								    (aref *pr* i5)))
							 (primep (+ (* (aref *pr* i5) p4s) p4)))
						    (let* ((p5 (aref *pr* i5))
							   (sum (+ p1 p2 p3 p4 p5)))
							  (format t "~s ~s ~s ~s ~s =~s\n" p1 p2 p3 p4 p5 sum)
							  (return-from euler60)
							  (when (or (null result)
								    (< sum result))
								(setf result sum))))))))))))
	result))

					 
; XLISP-PLUS, 13 seconds to find (first) answer. Takes forever if I let
; it run. Brute force (nested loops) with "tricks" -- no use of strings
; and knowledge that pn mod 3 must equal pm mod 3.

(in-package "USER")
