(require "factor")
(unless (find-package "EULER")
        (make-package "EULER" :use '("XLISP" "FACTOR")))
(use-package "EULER")
(in-package "EULER")
(export '(euler71 euler72 euler73 euler74
		  euler74a euler75 euler76
                  euler77 euler78 euler79 euler80))
(require "fact")

; We know from the Farey sequence that the next value between a/b and c/d is (a+c)/(b+d)

(defun euler71 (&aux (num 1) (den 1))
       (do ((a 2 (numerator (/ (+ a c) (+ b d))))
	    (b 5 (denominator (/ (+ a c) (+ b d)))) 
	    (c 3)
	    (d 7))
	   ((> b 1000000) (list num den))
	   (setf num a den b)))

; Made use of the information on the "Farey Sequence" to come up with a
; solution that took 190 msec. Not as efficient as it could be (double
; calculation of each (a+c)/(b+d)) but fast enough for the project.

; Time to do a phi sieve!

(defvar *phi*)

(defun euler72 ()
       (flet
	((makesieve (n)
		    (setf *phi* (make-array (1+ n)))
		    (dotimes (i (1+ n)) (setf (aref *phi* i) i)) ; initialize phi array to identity
		    (do ((i 2 (1+ i)))
			((> i n))
			(when (eql (aref *phi* i) i) ; unmarked - a prime
			      (do ((j i (+ j i))
				   (newterm (/ (1- i) i)))
				  ((> j n))
				  (setf (aref *phi* j)
					(* (aref *phi* j) newterm)))))))


	(makesieve 1000000)
	(1- (reduce #'+ *phi*)) ; subtract 1 for erroneous phi(1)
	))

; dldmem - 49.4 seconds, after expand 100 19.76 seconds xldmem - 5.31 seconds after expand 100

; Sum of phi(2) through phi(1,000,000). I made a sieve to calculate the phis. 5.3 seconds in XLISP-PLUS.

(defun euler73 ( &optional (limit 12000) &aux (result 0))
       (do ((d 5 (1+ d))) ; no fraction with d < 5 is within (1/3, 1/2)
           ((> d limit) result)
           (do ((n (1+ (floor d 3)) (1+ n)) ; look between 1/3 and 1/2
                (klimit (floor (1- d) 2)))
               ((> n klimit))
               (when (eql (gcd d n) 1)
                     (incf result)))))

; brute force, 9.3 seconds in XLISP-PLUS

; Make an array of factorials of digits

(defconstant +facts+ (let ((val nil))
		   (dotimes (i 10) (push (facti i) val))
		   (nreverse (coerce val 'array))))

; Make a hash table of looping values
(defconstant +loopy+ (make-hash-table :size 999997))

(mapc #'(lambda (v w) (setf (gethash v +loopy+) w))
      '(1 2 40585 145 169 363601 1454 871 45361 872 45362) ; looping value
      '(1 1 1     1   3   3      3    2   2     2   2)  ; length of chain
      )

(defun sumfactdigits (n)
       (reduce #'+
	       (map 'array
		    #'(lambda (v) (aref +facts+ (digit-char-p v)))
		    (princ-to-string n))))

(defun euler74 (&aux (result 0))
       (do ((i 69 (1+ i)))
	   ((eql i 1000000) result)
	   (do* ((inext i (sumfactdigits inext))
		 (count 0 (1+ count))
		 (hashval (gethash inext +loopy+) (gethash inext +loopy+)))
		(hashval
		 (when (eql 60 (+ count hashval))
		      (incf result))))))

(defun euler74a (&aux (result 0)) ; Increase size of loopy hash table by saving chain lengths
       (do ((i 69 (1+ i)))
	   ((eql i 1000000) result)
	   (do* ((inext (list i) (cons (sumfactdigits (car inext)) inext))
		 (count 0 (1+ count))
		 (hashval (gethash (car inext) +loopy+)
			  (gethash (car inext) +loopy+)))
		(hashval
		 (when (eql 60 (+ count hashval))
		      (incf result))
		 (mapc #'(lambda (v)
				 (setf (gethash v +loopy+) (+ count hashval))
				 (decf count))
		       (nreverse inext))))))
				 
; I created two tables, one for factorials up to 9 and the second a
; hash table of values and their chain lengths. The table was seeded
; with all the numbers in the problem statement plus 1,2, and 40585
; where are additional values with chain lengths of 1, like 145.
; 
; Then I just did a brute force solution going through sums of
; factorials of digits until I'd get a hit in the hash table.
; 
; The solution took 75 seconds, which is too long. So I modified the
; function to save all discovered chain lengths in the hash table during
; the processing. This new function, euler74a, gave the solution in 9.8
; seconds. 

(defun euler75 (&optional (maxval 1500000) &aux (res 0))
       (let ((result (make-array (1+ maxval) :initial-element 0))
	     (sqrtmaxval (floor (sqrt maxval))))
	    (do ((i 1 (+ i 2)))
		((> i sqrtmaxval))
		(do ((j 2 (+ j 2)))
		    ((> j (- sqrtmaxval i)))
		    (when (eql (gcd i j) 1)
			  (let ((circum (+ (abs (- (* j j) (* i i)))
					(* 2 i j)
					(* i i)
					(* j j))))
			       (do ((c circum (+ c circum)))
				   ((> c maxval))
				   (incf (aref result c)))))))
	    (map nil #'(lambda (v) (when (eql v 1) (incf res))) result)
	    res))

; I had to look up algorithms for Pythagorean triples, then I make an
; array to keep count of the quantity for each length and finally
; counted how many lengths had only one triple. 1.5 seconds in
; XLISP-PLUS.


; Used problem 31 solution with every integer from 1 to 99 as a coin

(defun euler76 (&optional (value 100) &aux coins)
       (labels ((euler3176 (money &optional (coins '(200 100 50 20 10 5 2 1)))
		       (if (null (cdr coins))
			   1	; only one choice if only one coin type left
			   (let ((ways 0))
				(do ((quant 0 (+ (car coins) quant))) ; take varying amounts of first coin in list
				    ((> quant money)) ; stop when too many coins
				    (incf ways (euler3176 (- money quant) (cdr coins))))
				ways))))

	     (dotimes (i (1- value)) (push (1+ i) coins))
	     (euler3176 value coins)))

; but this took over 5 minutes!

; Solve using code from problem 31.

(defun euler77 ()
       (labels ((euler3177 (money &optional (coins '(200 100 50 20 10 5 2 1)))
			 (if (null (cdr coins)) ; Only one coin left
			     (if (zerop (rem money (car coins))) 1 0) ; One choice if divisible by last coin
			     (let ((ways 0))
				  (do ((quant 0 (+ (car coins) quant))) ; take varying amounts of first coin in list
				      ((> quant money)) ; stop when too many coins
				      (incf ways (euler3177 (- money quant) (cdr coins))))
				  ways))))


	     (do* ((i 1 (1+ i)))
		  ((> (euler3177 i '(97 89 83 79 73 71 67 61 59 53 47 43 41 37 31 29 23 19 17 13 11 7 5 3 2))
		      5000) i))))

; Used problem 31's answer here to solve, much like I used it for
; problem 76. I did have to change the problem 31 code to allow for the
; residual value not being divisible by the smallest coin/prime. The
; original problem had a 1 penny coin that could always be used for a
; solution. 270 milliseconds.

(defun euler78 (&key (target 1000000) (max 100000))
       (let ((p (make-array max)))
            (setf (aref p 0) 1)
            (do ((n 1 (1+ n)))
                ((eql n max))
                (setf (aref p n) 0)
                (do* ((add 1 (+ 3 add))
                      (pent 1 (+ pent add))
                      (sign 1 (- sign))
                      (diff 1 (1+ diff)))
                     ((> pent n))
                     (incf (aref p n)
                           (* sign (+ (aref p (- n pent))
                                      (if (<= (+ pent diff) n)
                                          (aref p (- n pent diff))
                                          0))))
                     (setf (aref p n) (mod (aref p n) target)))
                (when (zerop (aref p n))
                      (format t "Result is - ~d\n" n)
                      (return-from euler78)))))

; The math necessary to get a solution fast enough was beyond my
; knowledge, so I rewrote a fast solution into XLISP-PLUS. Still, it took 17 seconds.

(defconstant +list+ '(
"319"
"680"
"180"
"690"
"129"
"620"
"762"
"689"
"762"
"318"
"368"
"710"
"720"
"710"
"629"
"168"
"160"
"689"
"716"
"731"
"736"
"729"
"316"
"729"
"729"
"710"
"769"
"290"
"719"
"680"
"318"
"389"
"162"
"289"
"162"
"718"
"729"
"319"
"790"
"680"
"890"
"362"
"319"
"760"
"316"
"729"
"380"
"319"
"728"
"716"))

(defun euler79 (&optional (samples +list+) &aux result)
       (let ((psamples (mapcar #'(lambda (s) (coerce s 'list)) samples)))
	   
	    (loop ; repeat until all processed
		  (let ((possibles nil))
		       ; Get list of possible first values
		       (mapc #'(lambda (s) (setf possibles (adjoin (car s) possibles)))
			     psamples)
		       ; Get rid of nil that appears in list when a choice is complete
		       (setf possibles (delete nil possibles))
		       ; Check if we are done
		       (when (null possibles) (return-from euler79 result))
		       ; Diagnostics
;		       (format t "Possibles ~s\n" possibles)
;		       (y-or-n-p)
		       ; Eliminate those that cannot be first
		       (mapc #'(lambda (s)
				       (setf possibles
					     (set-difference possibles
							     (intersection (cdr s) possibles))))
				psamples)
		       ; Must be at least one to choose from
		       (when (null possibles) (return-from euler79 "impossible"))
		       ; Report the value
;		       (format t "Next: ~a\n" possibles)
		       (setf result (append result possibles))
		       ; Eliminate it from the examples
		       (setf psamples
			     (mapcar #'(lambda (s) (reverse (set-difference s possibles)))
				     psamples))
		       ))))


; I did this by hand, but then went back to solve it in XLISP-PLUS since
; that's my personal goal. I converted the input to a list of lists of
; characters in each code. Then iterate over the following steps: 1.
; make a list of the unique first values of each code. 2. Prune that
; list of any values that appear beyond the first value of each code. 3.
; The remaining values in the list represent the next possible value(s)
; in the full passcode. 4. Remove these values from the list of lists.
; The iteration terminates as soon as there are no possible values,
; which is the same as there being no more values in the list of lists.
; Execution time is 500 microseconds.

(defun euler80 (&aux (result 0) (10to198 (expt 10 198)))
       (let ((psquares (make-hash-table)))
            (dotimes (i 10) (setf (gethash (* i i) psquares) t))
            (do ((i 2 (1+ i)))
                ((> i 99) result)
                (when (null (gethash i psquares))
                      (map nil
                           #'(lambda (c) (incf result (digit-char-p c)))
                            (princ-to-string (isqrt (* i 10to198))))))))

; I multiplied the value by 10e198 first and took an integer square
; root. I then added the digits in each one. A hash table of perfect
; squares was used to eliminated them from the calculation. 6.5 msec in
; XLISP-PLUS.

(in-package "USER")
